async function getItemFromChest(bot, itemToGet) {
    // Find all chests within a certain range (e.g., 32 blocks)
    const chests = bot.findBlocks({
        matching: block => block.name === "chest",
        maxDistance: 32,
        count: 10 // Limit to the first 10 chests found
    });

    if (chests.length === 0) {
        bot.chat("No chests found nearby.");
        console.log("No chests found nearby.");
        return;
    }

    for (const chestPosition of chests) {
        await moveToChest(bot, chestPosition);
        const chestBlock = bot.blockAt(chestPosition);
        const chest = await bot.openContainer(chestBlock);

        const itemByName = mcData.itemsByName[itemToGet];
        if (!itemByName) {
            bot.chat(`No item named ${itemToGet}`);
            console.log(`No item named ${itemToGet}`);
            continue;
        }

        const item = chest.findContainerItem(itemByName.id);
        if (!item) {
            bot.chat(`I don't see ${itemToGet} in this chest`);
            console.log(`I don't see ${itemToGet} in this chest`);
            continue;
        }
        try {
            await chest.withdraw(item.type, null, itemToGet[itemToGet]);
            bot.chat(`Successfully withdrew ${itemsToGet[itemToGet]} of ${itemToGet}`);
            console.log(`Successfully withdrew ${itemsToGet[itemToGet]} of ${itemToGet}`);
            break;
        } catch (err) {
            bot.chat(`Not enough ${itemToGet} in chest.`);
            console.log(`Not enough ${itemToGet} in chest.`);
        }
        
        await closeChest(bot, chestBlock);
    }
}

async function depositItemIntoChest(bot, itemToDeposit) {
    // Find all chests within a certain range (e.g., 32 blocks)
    const chests = bot.findBlocks({
        matching: block => block.name === "chest",
        maxDistance: 32,
        count: 10 // Limit to the first 10 chests found
    });

    if (chests.length === 0) {
        bot.chat("No chests found nearby.");
        console.log("No chests found nearby.");
        return;
    }

    const chestPosition = chests[0].position;
    await moveToChest(bot, chestPosition);
    const chestBlock = bot.blockAt(chestPosition);
    const chest = await bot.openContainer(chestBlock);
    const itemByName = mcData.itemsByName[itemToDeposit];
    if (!itemByName) {
        bot.chat(`No item named ${itemToDeposit}`);
        console.log(`No item named ${itemToDeposit}`);
    }
    const item = bot.inventory.findInventoryItem(itemByName.id);
    if (!item) {
        bot.chat(`No ${itemToDeposit} in inventory`);
        console.log(`No ${itemToDeposit} in inventory`);
    }
    try {
        await chest.deposit(item.type, null, itemToDeposit[itemToDeposit]);
        bot.chat(`Successfully deposited ${itemToDeposit[itemToDeposit]} of ${itemToDeposit}`);
        console.log(`Successfully deposited ${itemToDeposit[itemToDeposit]} of ${itemToDeposit}`);
    } catch (err) {
        bot.chat(`Not enough ${itemToDeposit} in inventory.`);
        console.log(`Not enough ${itemToDeposit} in inventory.`);
    }
    await closeChest(bot, chestBlock);
}

async function checkItemInsideChest(bot, chestPosition) {
    // return if chestPosition is not Vec3
    if (!(chestPosition instanceof Vec3)) {
        throw new Error(
            "chestPosition for depositItemIntoChest must be a Vec3"
        );
    }
    await moveToChest(bot, chestPosition);
    const chestBlock = bot.blockAt(chestPosition);
    await bot.openContainer(chestBlock);
    await closeChest(bot, chestBlock);
}

async function moveToChest(bot, chestPosition) {
    if (!(chestPosition instanceof Vec3)) {
        throw new Error(
            "chestPosition for depositItemIntoChest must be a Vec3"
        );
    }
    if (chestPosition.distanceTo(bot.entity.position) > 32) {
        bot.chat(
            `/tp ${chestPosition.x} ${chestPosition.y} ${chestPosition.z}`
        );
        await bot.waitForTicks(20);
    }
    const chestBlock = bot.blockAt(chestPosition);
    if (chestBlock.name !== "chest") {
        bot.emit("removeChest", chestPosition);
        throw new Error(
            `No chest at ${chestPosition}, it is ${chestBlock.name}`
        );
    }
    await bot.pathfinder.goto(
        new GoalLookAtBlock(chestBlock.position, bot.world, {})
    );
    return chestBlock;
}

async function listItemsInChest(bot, chestBlock) {
    const chest = await bot.openContainer(chestBlock);
    const items = chest.containerItems();
    if (items.length > 0) {
        const itemNames = items.reduce((acc, obj) => {
            if (acc[obj.name]) {
                acc[obj.name] += obj.count;
            } else {
                acc[obj.name] = obj.count;
            }
            return acc;
        }, {});
        bot.emit("closeChest", itemNames, chestBlock.position);
    } else {
        bot.emit("closeChest", {}, chestBlock.position);
    }
    return chest;
}

async function closeChest(bot, chestBlock) {
    try {
        const chest = await listItemsInChest(bot, chestBlock);
        await chest.close();
    } catch (err) {
        await bot.closeWindow(chestBlock);
    }
}

function itemByName(items, name) {
    for (let i = 0; i < items.length; ++i) {
        const item = items[i];
        if (item && item.name === name) return item;
    }
    return null;
}
